/*
 * SFTP protocol delegating to sftp_wrap
 * Copyright (c) 2024 MXPlayer Android
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "libavutil/avstring.h"
#include "libavutil/opt.h"
#include "avformat.h"
#include "internal.h"
#include "url.h"
#include "sftp_wrap.h"

typedef struct SFTPContext {
    const AVClass *class;
    char *private_key;
} SFTPContext;

#define OFFSET(x) offsetof(SFTPContext, x)
#define D AV_OPT_FLAG_DECODING_PARAM
#define E AV_OPT_FLAG_ENCODING_PARAM

static const AVOption options[] = {
    {"private_key", "path to private key file for authentication", OFFSET(private_key), AV_OPT_TYPE_STRING, {.str = NULL}, 0, 0, D|E },
    {NULL}
};

static const AVClass sftp_context_class = {
    .class_name = "sftp",
    .item_name  = av_default_item_name,
    .option     = options,
    .version    = LIBAVUTIL_VERSION_INT,
};

static int sftp_wrapper_open(URLContext *h, const char *url, int flags)
{
    SFTPContext *s = h->priv_data;
    return sftp_open(h, url, flags, s->private_key);
}

static int64_t sftp_wrapper_seek(URLContext *h, int64_t pos, int whence)
{
    return sftp_seek(h, pos, whence);
}

static int sftp_wrapper_read(URLContext *h, unsigned char *buf, int size)
{
    return sftp_read(h, buf, size);
}

static int sftp_wrapper_write(URLContext *h, const unsigned char *buf, int size)
{
    return sftp_write(h, buf, size);
}

static int sftp_wrapper_delete(URLContext *h)
{
    return sftp_delete(h);
}

static int sftp_wrapper_move(URLContext *h_src, URLContext *h_dst)
{
    return sftp_move(h_src, h_dst);
}

static int sftp_wrapper_close(URLContext *h)
{
    return sftp_close(h);
}

static int sftp_wrapper_open_dir(URLContext *h)
{
    return sftp_open_dir(h);
}

static int sftp_wrapper_read_dir(URLContext *h, AVIODirEntry **next)
{
    return sftp_read_dir(h, (void**)next);
}

static int sftp_wrapper_close_dir(URLContext *h)
{
    return sftp_close_dir(h);
}

const URLProtocol ff_sftp_protocol = {
    .name                = "sftp",
    .url_open            = sftp_wrapper_open,
    .url_read            = sftp_wrapper_read,
    .url_write           = sftp_wrapper_write,
    .url_seek            = sftp_wrapper_seek,
    .url_close           = sftp_wrapper_close,
    .url_open_dir        = sftp_wrapper_open_dir,
    .url_read_dir        = sftp_wrapper_read_dir,
    .url_close_dir       = sftp_wrapper_close_dir,
    .url_delete          = sftp_wrapper_delete,
    .url_move            = sftp_wrapper_move,
    .priv_data_size      = sizeof(SFTPContext),
    .priv_data_class     = &sftp_context_class,
    .flags               = URL_PROTOCOL_FLAG_NETWORK,
};
