/*
 * FTPS (FTP over TLS/SSL) protocol
 * Copyright (c) 2024 MXPlayer Android
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "libavutil/avstring.h"
#include "libavutil/opt.h"
#include "avformat.h"
#include "internal.h"
#include "url.h"
#include "ftps_wrap.h"

typedef struct FTPSContext {
    const AVClass *class;
    int implicit;
    int passive_mode;
    char *encoding;
} FTPSContext;

#define OFFSET(x) offsetof(FTPSContext, x)
#define D AV_OPT_FLAG_DECODING_PARAM
#define E AV_OPT_FLAG_ENCODING_PARAM

static const AVOption options[] = {
    {"implicit", "use implicit FTPS (port 990, default 0)", OFFSET(implicit), AV_OPT_TYPE_BOOL, {.i64 = 0}, 0, 1, D|E },
    {"passive", "use passive mode for data connections (default 1)", OFFSET(passive_mode), AV_OPT_TYPE_BOOL, {.i64 = 1}, 0, 1, D|E },
    {"encoding", "character encoding for filenames", OFFSET(encoding), AV_OPT_TYPE_STRING, {.str = "UTF-8"}, 0, 0, D|E },
    {NULL}
};

static const AVClass ftps_context_class = {
    .class_name = "ftps",
    .item_name  = av_default_item_name,
    .option     = options,
    .version    = LIBAVUTIL_VERSION_INT,
};

static int ftps_wrapper_open(URLContext *h, const char *url, int flags)
{
    FTPSContext *s = h->priv_data;
    return ftps_open(h, url, flags, s->implicit, s->passive_mode, s->encoding);
}

static int64_t ftps_wrapper_seek(URLContext *h, int64_t pos, int whence)
{
    return ftps_seek(h, pos, whence);
}

static int ftps_wrapper_read(URLContext *h, unsigned char *buf, int size)
{
    return ftps_read(h, buf, size);
}

static int ftps_wrapper_write(URLContext *h, const unsigned char *buf, int size)
{
    return ftps_write(h, buf, size);
}

static int ftps_wrapper_delete(URLContext *h)
{
    return ftps_delete(h);
}

static int ftps_wrapper_move(URLContext *h_src, URLContext *h_dst)
{
    return ftps_move(h_src, h_dst);
}

static int ftps_wrapper_close(URLContext *h)
{
    return ftps_close(h);
}

static int ftps_wrapper_open_dir(URLContext *h)
{
    return ftps_open_dir(h);
}

static int ftps_wrapper_read_dir(URLContext *h, AVIODirEntry **next)
{
    return ftps_read_dir(h, (void**)next);
}

static int ftps_wrapper_close_dir(URLContext *h)
{
    return ftps_close_dir(h);
}

const URLProtocol ff_ftps_protocol = {
    .name                = "ftps",
    .url_open            = ftps_wrapper_open,
    .url_read            = ftps_wrapper_read,
    .url_write           = ftps_wrapper_write,
    .url_seek            = ftps_wrapper_seek,
    .url_close           = ftps_wrapper_close,
    .url_open_dir        = ftps_wrapper_open_dir,
    .url_read_dir        = ftps_wrapper_read_dir,
    .url_close_dir       = ftps_wrapper_close_dir,
    .url_delete          = ftps_wrapper_delete,
    .url_move            = ftps_wrapper_move,
    .priv_data_size      = sizeof(FTPSContext),
    .priv_data_class     = &ftps_context_class,
    .flags               = URL_PROTOCOL_FLAG_NETWORK,
};
