/*
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "dirac_vlc.h"

enum {
    /* Next byte contains an exactly aligned start to a new symbol (even bit) */
    STATE_START  = 0,
    /* Next byte should end the current value on an odd bit */
    STATE_FOLLOW = 256,
    /* Byte is completely data and doesn't end nor start a value */
    STATE_DATA   = 512,
    /* Byte has the current value's sign bit and starts a new value */
    STATE_SIGN   = 768,
};

/* Exactly 128 bits */
typedef struct LUTState {
    int16_t   val0;      /* Bits to which to add after applying preshift */
    int16_t   val1;
    int16_t   val2;
    int16_t   val3;
    int16_t   val4;
    uint8_t   val0_bits; /* The size of val0 in bits */
    int8_t    sign;      /* Sign of the current value (0 == zero the value) */
    int8_t    num;       /* Number of values in this byte */
    uint8_t   val;       /* Init value in case current value was terminated */
    uint16_t  state;     /* Expected state for the next byte */
} LUTState;

const DECLARE_ALIGNED(32, LUTState, ff_dirac_golomb_lut)[1024] = {
    { +16,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +17,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    {  +8,  0,  0,  0,  0, 4, +1, 1,  0,  STATE_START },
    {  +8,  0,  0,  0,  0, 4, -1, 1,  0,  STATE_START },
    { +18,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +19,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    {  +9,  0,  0,  0,  0, 4, +1, 1,  0,  STATE_START },
    {  +9,  0,  0,  0,  0, 4, -1, 1,  0,  STATE_START },
    {  +4,  0,  0,  0,  0, 3, +1, 1,  2, STATE_FOLLOW },
    {  +4,  0,  0,  0,  0, 3, +1, 1,  3, STATE_FOLLOW },
    {  +4,  0,  0,  0,  0, 3, +1, 2,  1,   STATE_DATA },
    {  +4,  0,  0,  0,  0, 3, +1, 3,  0,  STATE_START },
    {  +4,  0,  0,  0,  0, 3, -1, 1,  2, STATE_FOLLOW },
    {  +4,  0,  0,  0,  0, 3, -1, 1,  3, STATE_FOLLOW },
    {  +4,  0,  0,  0,  0, 3, -1, 2,  1,   STATE_DATA },
    {  +4,  0,  0,  0,  0, 3, -1, 3,  0,  STATE_START },
    { +20,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +21,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +10,  0,  0,  0,  0, 4, +1, 1,  0,  STATE_START },
    { +10,  0,  0,  0,  0, 4, -1, 1,  0,  STATE_START },
    { +22,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +23,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +11,  0,  0,  0,  0, 4, +1, 1,  0,  STATE_START },
    { +11,  0,  0,  0,  0, 4, -1, 1,  0,  STATE_START },
    {  +5,  0,  0,  0,  0, 3, +1, 1,  2, STATE_FOLLOW },
    {  +5,  0,  0,  0,  0, 3, +1, 1,  3, STATE_FOLLOW },
    {  +5,  0,  0,  0,  0, 3, +1, 2,  1,   STATE_DATA },
    {  +5,  0,  0,  0,  0, 3, +1, 3,  0,  STATE_START },
    {  +5,  0,  0,  0,  0, 3, -1, 1,  2, STATE_FOLLOW },
    {  +5,  0,  0,  0,  0, 3, -1, 1,  3, STATE_FOLLOW },
    {  +5,  0,  0,  0,  0, 3, -1, 2,  1,   STATE_DATA },
    {  +5,  0,  0,  0,  0, 3, -1, 3,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  4, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  5, STATE_FOLLOW },
    {  +2, +1,  0,  0,  0, 2, +1, 2,  0,  STATE_START },
    {  +2, -1,  0,  0,  0, 2, +1, 2,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  6, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  7, STATE_FOLLOW },
    {  +2, +2,  0,  0,  0, 2, +1, 2,  0,  STATE_START },
    {  +2, -2,  0,  0,  0, 2, +1, 2,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, +1, 2,  2,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, +1, 2,  2,   STATE_SIGN },
    {  +2,  0,  0,  0,  0, 2, +1, 2,  3,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, +1, 2,  3,   STATE_SIGN },
    {  +2,  0,  0,  0,  0, 2, +1, 3,  2, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, +1, 3,  3, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, +1, 4,  1,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, +1, 5,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  4, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  5, STATE_FOLLOW },
    {  +2, +1,  0,  0,  0, 2, -1, 2,  0,  STATE_START },
    {  +2, -1,  0,  0,  0, 2, -1, 2,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  6, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  7, STATE_FOLLOW },
    {  +2, +2,  0,  0,  0, 2, -1, 2,  0,  STATE_START },
    {  +2, -2,  0,  0,  0, 2, -1, 2,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, -1, 2,  2,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, -1, 2,  2,   STATE_SIGN },
    {  +2,  0,  0,  0,  0, 2, -1, 2,  3,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, -1, 2,  3,   STATE_SIGN },
    {  +2,  0,  0,  0,  0, 2, -1, 3,  2, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, -1, 3,  3, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, -1, 4,  1,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, -1, 5,  0,  STATE_START },
    { +24,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +25,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +12,  0,  0,  0,  0, 4, +1, 1,  0,  STATE_START },
    { +12,  0,  0,  0,  0, 4, -1, 1,  0,  STATE_START },
    { +26,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +27,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +13,  0,  0,  0,  0, 4, +1, 1,  0,  STATE_START },
    { +13,  0,  0,  0,  0, 4, -1, 1,  0,  STATE_START },
    {  +6,  0,  0,  0,  0, 3, +1, 1,  2, STATE_FOLLOW },
    {  +6,  0,  0,  0,  0, 3, +1, 1,  3, STATE_FOLLOW },
    {  +6,  0,  0,  0,  0, 3, +1, 2,  1,   STATE_DATA },
    {  +6,  0,  0,  0,  0, 3, +1, 3,  0,  STATE_START },
    {  +6,  0,  0,  0,  0, 3, -1, 1,  2, STATE_FOLLOW },
    {  +6,  0,  0,  0,  0, 3, -1, 1,  3, STATE_FOLLOW },
    {  +6,  0,  0,  0,  0, 3, -1, 2,  1,   STATE_DATA },
    {  +6,  0,  0,  0,  0, 3, -1, 3,  0,  STATE_START },
    { +28,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +29,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +14,  0,  0,  0,  0, 4, +1, 1,  0,  STATE_START },
    { +14,  0,  0,  0,  0, 4, -1, 1,  0,  STATE_START },
    { +30,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +31,  0,  0,  0,  0, 5, +1, 0,  0, STATE_FOLLOW },
    { +15,  0,  0,  0,  0, 4, +1, 1,  0,  STATE_START },
    { +15,  0,  0,  0,  0, 4, -1, 1,  0,  STATE_START },
    {  +7,  0,  0,  0,  0, 3, +1, 1,  2, STATE_FOLLOW },
    {  +7,  0,  0,  0,  0, 3, +1, 1,  3, STATE_FOLLOW },
    {  +7,  0,  0,  0,  0, 3, +1, 2,  1,   STATE_DATA },
    {  +7,  0,  0,  0,  0, 3, +1, 3,  0,  STATE_START },
    {  +7,  0,  0,  0,  0, 3, -1, 1,  2, STATE_FOLLOW },
    {  +7,  0,  0,  0,  0, 3, -1, 1,  3, STATE_FOLLOW },
    {  +7,  0,  0,  0,  0, 3, -1, 2,  1,   STATE_DATA },
    {  +7,  0,  0,  0,  0, 3, -1, 3,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  4, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  5, STATE_FOLLOW },
    {  +3, +1,  0,  0,  0, 2, +1, 2,  0,  STATE_START },
    {  +3, -1,  0,  0,  0, 2, +1, 2,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  6, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  7, STATE_FOLLOW },
    {  +3, +2,  0,  0,  0, 2, +1, 2,  0,  STATE_START },
    {  +3, -2,  0,  0,  0, 2, +1, 2,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, +1, 2,  2,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, +1, 2,  2,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 2, +1, 2,  3,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, +1, 2,  3,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 2, +1, 3,  2, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, +1, 3,  3, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, +1, 4,  1,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, +1, 5,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  4, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  5, STATE_FOLLOW },
    {  +3, +1,  0,  0,  0, 2, -1, 2,  0,  STATE_START },
    {  +3, -1,  0,  0,  0, 2, -1, 2,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  6, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  7, STATE_FOLLOW },
    {  +3, +2,  0,  0,  0, 2, -1, 2,  0,  STATE_START },
    {  +3, -2,  0,  0,  0, 2, -1, 2,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, -1, 2,  2,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, -1, 2,  2,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 2, -1, 2,  3,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, -1, 2,  3,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 2, -1, 3,  2, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, -1, 3,  3, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, -1, 4,  1,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 1,  8,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 1,  8,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0,  0, 1,  9,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 1,  9,   STATE_SIGN },
    {   0, +3,  0,  0,  0, 0,  0, 2,  1,   STATE_DATA },
    {   0, +3,  0,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0, -3,  0,  0,  0, 0,  0, 2,  1,   STATE_DATA },
    {   0, -3,  0,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 1, 10,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 1, 10,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0,  0, 1, 11,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 1, 11,   STATE_SIGN },
    {   0, +4,  0,  0,  0, 0,  0, 2,  1,   STATE_DATA },
    {   0, +4,  0,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0, -4,  0,  0,  0, 0,  0, 2,  1,   STATE_DATA },
    {   0, -4,  0,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0, +1,  0,  0,  0, 0,  0, 2,  2,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0,  0, 2,  2,   STATE_SIGN },
    {   0, +1,  0,  0,  0, 0,  0, 2,  3,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0,  0, 2,  3,   STATE_SIGN },
    {   0, +1,  0,  0,  0, 0,  0, 3,  2, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0,  0, 3,  3, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0,  0, 5,  0,  STATE_START },
    {   0, -1,  0,  0,  0, 0,  0, 2,  2,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0,  0, 2,  2,   STATE_SIGN },
    {   0, -1,  0,  0,  0, 0,  0, 2,  3,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0,  0, 2,  3,   STATE_SIGN },
    {   0, -1,  0,  0,  0, 0,  0, 3,  2, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0,  0, 3,  3, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 1, 12,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 1, 12,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0,  0, 1, 13,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 1, 13,   STATE_SIGN },
    {   0, +5,  0,  0,  0, 0,  0, 2,  1,   STATE_DATA },
    {   0, +5,  0,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0, -5,  0,  0,  0, 0,  0, 2,  1,   STATE_DATA },
    {   0, -5,  0,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 1, 14,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 1, 14,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0,  0, 1, 15,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 1, 15,   STATE_SIGN },
    {   0, +6,  0,  0,  0, 0,  0, 2,  1,   STATE_DATA },
    {   0, +6,  0,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0, -6,  0,  0,  0, 0,  0, 2,  1,   STATE_DATA },
    {   0, -6,  0,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0, +2,  0,  0,  0, 0,  0, 2,  2,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0,  0, 2,  2,   STATE_SIGN },
    {   0, +2,  0,  0,  0, 0,  0, 2,  3,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0,  0, 2,  3,   STATE_SIGN },
    {   0, +2,  0,  0,  0, 0,  0, 3,  2, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0,  0, 3,  3, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0,  0, 5,  0,  STATE_START },
    {   0, -2,  0,  0,  0, 0,  0, 2,  2,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0,  0, 2,  2,   STATE_SIGN },
    {   0, -2,  0,  0,  0, 0,  0, 2,  3,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0,  0, 2,  3,   STATE_SIGN },
    {   0, -2,  0,  0,  0, 0,  0, 3,  2, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0,  0, 3,  3, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 2,  8, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0,  0, 2,  9, STATE_FOLLOW },
    {   0,  0, +3,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0, -3,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 2, 10, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0,  0, 2, 11, STATE_FOLLOW },
    {   0,  0, +4,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0, -4,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0, +1,  0,  0, 0,  0, 3,  2, STATE_FOLLOW },
    {   0,  0, +1,  0,  0, 0,  0, 3,  3, STATE_FOLLOW },
    {   0,  0, +1,  0,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0,  0, +1,  0,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0, -1,  0,  0, 0,  0, 3,  2, STATE_FOLLOW },
    {   0,  0, -1,  0,  0, 0,  0, 3,  3, STATE_FOLLOW },
    {   0,  0, -1,  0,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0,  0, -1,  0,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 2, 12, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0,  0, 2, 13, STATE_FOLLOW },
    {   0,  0, +5,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0, -5,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 2, 14, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0,  0, 2, 15, STATE_FOLLOW },
    {   0,  0, +6,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0, -6,  0,  0, 0,  0, 3,  0,  STATE_START },
    {   0,  0, +2,  0,  0, 0,  0, 3,  2, STATE_FOLLOW },
    {   0,  0, +2,  0,  0, 0,  0, 3,  3, STATE_FOLLOW },
    {   0,  0, +2,  0,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0,  0, +2,  0,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0, -2,  0,  0, 0,  0, 3,  2, STATE_FOLLOW },
    {   0,  0, -2,  0,  0, 0,  0, 3,  3, STATE_FOLLOW },
    {   0,  0, -2,  0,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0,  0, -2,  0,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 3,  4,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 3,  4,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0,  0, 3,  5,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 3,  5,   STATE_SIGN },
    {   0,  0,  0, +1,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0,  0,  0, +1,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0, -1,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0,  0,  0, -1,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 3,  6,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 3,  6,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0,  0, 3,  7,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 3,  7,   STATE_SIGN },
    {   0,  0,  0, +2,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0,  0,  0, +2,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0, -2,  0, 0,  0, 4,  1,   STATE_DATA },
    {   0,  0,  0, -2,  0, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 4,  4, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0,  0, 4,  5, STATE_FOLLOW },
    {   0,  0,  0,  0, +1, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0, -1, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 4,  6, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0,  0, 4,  7, STATE_FOLLOW },
    {   0,  0,  0,  0, +2, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0, -2, 0,  0, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0,  0, 5,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 5,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0,  0, 5,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 5,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0,  0, 6,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0,  0, 6,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0,  0, 7,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0,  0, 8,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 3, +1, 1,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 3, -1, 1,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 3, +1, 1,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 3, -1, 1,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 2, +1, 1,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 2, +1, 1,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 2, +1, 2,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 2, +1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 2, -1, 1,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 2, -1, 1,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 2, -1, 2,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 2, -1, 3,  0,  STATE_START },
    {  +4,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +5,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 3, +1, 1,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 3, -1, 1,  0,  STATE_START },
    {  +6,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +7,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 3, +1, 1,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 3, -1, 1,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 2, +1, 1,  2, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 2, +1, 1,  3, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 2, +1, 2,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 2, +1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 2, -1, 1,  2, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 2, -1, 1,  3, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 2, -1, 2,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 2, -1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, +1, 1,  4, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, +1, 1,  5, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 1, +1, 2,  0,  STATE_START },
    {   0, -1,  0,  0,  0, 1, +1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, +1, 1,  6, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, +1, 1,  7, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 1, +1, 2,  0,  STATE_START },
    {   0, -2,  0,  0,  0, 1, +1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, +1, 2,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 2,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, +1, 2,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 2,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, +1, 3,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, +1, 3,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, +1, 4,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, -1, 1,  4, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, -1, 1,  5, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 1, -1, 2,  0,  STATE_START },
    {   0, -1,  0,  0,  0, 1, -1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, -1, 1,  6, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, -1, 1,  7, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 1, -1, 2,  0,  STATE_START },
    {   0, -2,  0,  0,  0, 1, -1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, -1, 2,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 2,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, -1, 2,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 2,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, -1, 3,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, -1, 3,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, -1, 4,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 5,  0,  STATE_START },
    {  +8,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +9,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +4,  0,  0,  0,  0, 3, +1, 1,  0,  STATE_START },
    {  +4,  0,  0,  0,  0, 3, -1, 1,  0,  STATE_START },
    { +10,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    { +11,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +5,  0,  0,  0,  0, 3, +1, 1,  0,  STATE_START },
    {  +5,  0,  0,  0,  0, 3, -1, 1,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  2, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  3, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, +1, 2,  1,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, +1, 3,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  2, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  3, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, -1, 2,  1,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, -1, 3,  0,  STATE_START },
    { +12,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    { +13,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +6,  0,  0,  0,  0, 3, +1, 1,  0,  STATE_START },
    {  +6,  0,  0,  0,  0, 3, -1, 1,  0,  STATE_START },
    { +14,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    { +15,  0,  0,  0,  0, 4, +1, 0,  0, STATE_FOLLOW },
    {  +7,  0,  0,  0,  0, 3, +1, 1,  0,  STATE_START },
    {  +7,  0,  0,  0,  0, 3, -1, 1,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  2, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  3, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, +1, 2,  1,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, +1, 3,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  2, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  3, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, -1, 2,  1,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, -1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  4, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  5, STATE_FOLLOW },
    {  +1, +1,  0,  0,  0, 1, +1, 2,  0,  STATE_START },
    {  +1, -1,  0,  0,  0, 1, +1, 2,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  6, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  7, STATE_FOLLOW },
    {  +1, +2,  0,  0,  0, 1, +1, 2,  0,  STATE_START },
    {  +1, -2,  0,  0,  0, 1, +1, 2,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, +1, 2,  2,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 2,  2,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, +1, 2,  3,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 2,  3,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, +1, 3,  2, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, +1, 3,  3, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, +1, 4,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 5,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  4, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  5, STATE_FOLLOW },
    {  +1, +1,  0,  0,  0, 1, -1, 2,  0,  STATE_START },
    {  +1, -1,  0,  0,  0, 1, -1, 2,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  6, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  7, STATE_FOLLOW },
    {  +1, +2,  0,  0,  0, 1, -1, 2,  0,  STATE_START },
    {  +1, -2,  0,  0,  0, 1, -1, 2,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, -1, 2,  2,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 2,  2,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, -1, 2,  3,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 2,  3,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, -1, 3,  2, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, -1, 3,  3, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, -1, 4,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 1,  8, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 1,  9, STATE_FOLLOW },
    {   0, +3,  0,  0,  0, 0, +1, 2,  0,  STATE_START },
    {   0, -3,  0,  0,  0, 0, +1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 1, 10, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 1, 11, STATE_FOLLOW },
    {   0, +4,  0,  0,  0, 0, +1, 2,  0,  STATE_START },
    {   0, -4,  0,  0,  0, 0, +1, 2,  0,  STATE_START },
    {   0, +1,  0,  0,  0, 0, +1, 2,  2, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0, +1, 2,  3, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0, +1, 3,  1,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0, +1, 4,  0,  STATE_START },
    {   0, -1,  0,  0,  0, 0, +1, 2,  2, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0, +1, 2,  3, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0, +1, 3,  1,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 1, 12, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 1, 13, STATE_FOLLOW },
    {   0, +5,  0,  0,  0, 0, +1, 2,  0,  STATE_START },
    {   0, -5,  0,  0,  0, 0, +1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 1, 14, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 1, 15, STATE_FOLLOW },
    {   0, +6,  0,  0,  0, 0, +1, 2,  0,  STATE_START },
    {   0, -6,  0,  0,  0, 0, +1, 2,  0,  STATE_START },
    {   0, +2,  0,  0,  0, 0, +1, 2,  2, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0, +1, 2,  3, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0, +1, 3,  1,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0, +1, 4,  0,  STATE_START },
    {   0, -2,  0,  0,  0, 0, +1, 2,  2, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0, +1, 2,  3, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0, +1, 3,  1,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 2,  4,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 2,  4,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 2,  5,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 2,  5,   STATE_SIGN },
    {   0,  0, +1,  0,  0, 0, +1, 3,  1,   STATE_DATA },
    {   0,  0, +1,  0,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0, -1,  0,  0, 0, +1, 3,  1,   STATE_DATA },
    {   0,  0, -1,  0,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 2,  6,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 2,  6,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 2,  7,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 2,  7,   STATE_SIGN },
    {   0,  0, +2,  0,  0, 0, +1, 3,  1,   STATE_DATA },
    {   0,  0, +2,  0,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0, -2,  0,  0, 0, +1, 3,  1,   STATE_DATA },
    {   0,  0, -2,  0,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 3,  4, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 3,  5, STATE_FOLLOW },
    {   0,  0,  0, +1,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0,  0, -1,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 3,  6, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 3,  7, STATE_FOLLOW },
    {   0,  0,  0, +2,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0,  0, -2,  0, 0, +1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 4,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 4,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 4,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 4,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 5,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 5,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 6,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 7,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 1,  8, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 1,  9, STATE_FOLLOW },
    {   0, +3,  0,  0,  0, 0, -1, 2,  0,  STATE_START },
    {   0, -3,  0,  0,  0, 0, -1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 1, 10, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 1, 11, STATE_FOLLOW },
    {   0, +4,  0,  0,  0, 0, -1, 2,  0,  STATE_START },
    {   0, -4,  0,  0,  0, 0, -1, 2,  0,  STATE_START },
    {   0, +1,  0,  0,  0, 0, -1, 2,  2, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0, -1, 2,  3, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0, -1, 3,  1,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0, -1, 4,  0,  STATE_START },
    {   0, -1,  0,  0,  0, 0, -1, 2,  2, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0, -1, 2,  3, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0, -1, 3,  1,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 1, 12, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 1, 13, STATE_FOLLOW },
    {   0, +5,  0,  0,  0, 0, -1, 2,  0,  STATE_START },
    {   0, -5,  0,  0,  0, 0, -1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 1, 14, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 1, 15, STATE_FOLLOW },
    {   0, +6,  0,  0,  0, 0, -1, 2,  0,  STATE_START },
    {   0, -6,  0,  0,  0, 0, -1, 2,  0,  STATE_START },
    {   0, +2,  0,  0,  0, 0, -1, 2,  2, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0, -1, 2,  3, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0, -1, 3,  1,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0, -1, 4,  0,  STATE_START },
    {   0, -2,  0,  0,  0, 0, -1, 2,  2, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0, -1, 2,  3, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0, -1, 3,  1,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 2,  4,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 2,  4,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 2,  5,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 2,  5,   STATE_SIGN },
    {   0,  0, +1,  0,  0, 0, -1, 3,  1,   STATE_DATA },
    {   0,  0, +1,  0,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0, -1,  0,  0, 0, -1, 3,  1,   STATE_DATA },
    {   0,  0, -1,  0,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 2,  6,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 2,  6,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 2,  7,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 2,  7,   STATE_SIGN },
    {   0,  0, +2,  0,  0, 0, -1, 3,  1,   STATE_DATA },
    {   0,  0, +2,  0,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0, -2,  0,  0, 0, -1, 3,  1,   STATE_DATA },
    {   0,  0, -2,  0,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 3,  4, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 3,  5, STATE_FOLLOW },
    {   0,  0,  0, +1,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0,  0, -1,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 3,  6, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 3,  7, STATE_FOLLOW },
    {   0,  0,  0, +2,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0,  0, -2,  0, 0, -1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 4,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 4,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 4,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 4,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 5,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 5,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 6,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 7,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {   0,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 3, +1, 1,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 3, +1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 3, -1, 1,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 3, -1, 2,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 3, +1, 1,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 3, +1, 2,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 3, -1, 1,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 3, -1, 2,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 2, +1, 1,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 2, +1, 1,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 2, +1, 1,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 2, +1, 1,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 2, +1, 2,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 2, +1, 2,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 2, +1, 3,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 2, +1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 2, -1, 1,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 2, -1, 1,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 2, -1, 1,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 2, -1, 1,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 2, -1, 2,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 2, -1, 2,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 2, -1, 3,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 2, -1, 4,  0,  STATE_START },
    {  +4,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {  +4,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +5,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {  +5,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +2,  0,  0,  0,  0, 3, +1, 1,  1,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 3, +1, 2,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 3, -1, 1,  1,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 3, -1, 2,  0,  STATE_START },
    {  +6,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {  +6,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +7,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {  +7,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 3, +1, 1,  1,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 3, +1, 2,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 3, -1, 1,  1,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 3, -1, 2,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 2, +1, 1,  2,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 2, +1, 1,  2,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 2, +1, 1,  3,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 2, +1, 1,  3,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 2, +1, 2,  2, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 2, +1, 2,  3, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 2, +1, 3,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 2, +1, 4,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 2, -1, 1,  2,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 2, -1, 1,  2,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 2, -1, 1,  3,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 2, -1, 1,  3,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 2, -1, 2,  2, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 2, -1, 2,  3, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 2, -1, 3,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 2, -1, 4,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, +1, 1,  4,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 1,  4,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, +1, 1,  5,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 1,  5,   STATE_SIGN },
    {   0, +1,  0,  0,  0, 1, +1, 2,  1,   STATE_DATA },
    {   0, +1,  0,  0,  0, 1, +1, 3,  0,  STATE_START },
    {   0, -1,  0,  0,  0, 1, +1, 2,  1,   STATE_DATA },
    {   0, -1,  0,  0,  0, 1, +1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, +1, 1,  6,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 1,  6,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, +1, 1,  7,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 1,  7,   STATE_SIGN },
    {   0, +2,  0,  0,  0, 1, +1, 2,  1,   STATE_DATA },
    {   0, +2,  0,  0,  0, 1, +1, 3,  0,  STATE_START },
    {   0, -2,  0,  0,  0, 1, +1, 2,  1,   STATE_DATA },
    {   0, -2,  0,  0,  0, 1, +1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, +1, 2,  4, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, +1, 2,  5, STATE_FOLLOW },
    {   0,  0, +1,  0,  0, 1, +1, 3,  0,  STATE_START },
    {   0,  0, -1,  0,  0, 1, +1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, +1, 2,  6, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, +1, 2,  7, STATE_FOLLOW },
    {   0,  0, +2,  0,  0, 1, +1, 3,  0,  STATE_START },
    {   0,  0, -2,  0,  0, 1, +1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, +1, 3,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 3,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, +1, 3,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 3,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, +1, 4,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, +1, 4,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, +1, 5,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, +1, 6,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, -1, 1,  4,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 1,  4,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, -1, 1,  5,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 1,  5,   STATE_SIGN },
    {   0, +1,  0,  0,  0, 1, -1, 2,  1,   STATE_DATA },
    {   0, +1,  0,  0,  0, 1, -1, 3,  0,  STATE_START },
    {   0, -1,  0,  0,  0, 1, -1, 2,  1,   STATE_DATA },
    {   0, -1,  0,  0,  0, 1, -1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, -1, 1,  6,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 1,  6,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, -1, 1,  7,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 1,  7,   STATE_SIGN },
    {   0, +2,  0,  0,  0, 1, -1, 2,  1,   STATE_DATA },
    {   0, +2,  0,  0,  0, 1, -1, 3,  0,  STATE_START },
    {   0, -2,  0,  0,  0, 1, -1, 2,  1,   STATE_DATA },
    {   0, -2,  0,  0,  0, 1, -1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, -1, 2,  4, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, -1, 2,  5, STATE_FOLLOW },
    {   0,  0, +1,  0,  0, 1, -1, 3,  0,  STATE_START },
    {   0,  0, -1,  0,  0, 1, -1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, -1, 2,  6, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, -1, 2,  7, STATE_FOLLOW },
    {   0,  0, +2,  0,  0, 1, -1, 3,  0,  STATE_START },
    {   0,  0, -2,  0,  0, 1, -1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 1, -1, 3,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 3,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, -1, 3,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 3,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 1, -1, 4,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, -1, 4,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 1, -1, 5,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 1, -1, 6,  0,  STATE_START },
    {  +8,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {  +8,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +9,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    {  +9,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +4,  0,  0,  0,  0, 3, +1, 1,  1,   STATE_DATA },
    {  +4,  0,  0,  0,  0, 3, +1, 2,  0,  STATE_START },
    {  +4,  0,  0,  0,  0, 3, -1, 1,  1,   STATE_DATA },
    {  +4,  0,  0,  0,  0, 3, -1, 2,  0,  STATE_START },
    { +10,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    { +10,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    { +11,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    { +11,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +5,  0,  0,  0,  0, 3, +1, 1,  1,   STATE_DATA },
    {  +5,  0,  0,  0,  0, 3, +1, 2,  0,  STATE_START },
    {  +5,  0,  0,  0,  0, 3, -1, 1,  1,   STATE_DATA },
    {  +5,  0,  0,  0,  0, 3, -1, 2,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  2,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  2,   STATE_SIGN },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  3,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, +1, 1,  3,   STATE_SIGN },
    {  +2,  0,  0,  0,  0, 2, +1, 2,  2, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, +1, 2,  3, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, +1, 3,  1,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, +1, 4,  0,  STATE_START },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  2,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  2,   STATE_SIGN },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  3,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, -1, 1,  3,   STATE_SIGN },
    {  +2,  0,  0,  0,  0, 2, -1, 2,  2, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, -1, 2,  3, STATE_FOLLOW },
    {  +2,  0,  0,  0,  0, 2, -1, 3,  1,   STATE_DATA },
    {  +2,  0,  0,  0,  0, 2, -1, 4,  0,  STATE_START },
    { +12,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    { +12,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    { +13,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    { +13,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +6,  0,  0,  0,  0, 3, +1, 1,  1,   STATE_DATA },
    {  +6,  0,  0,  0,  0, 3, +1, 2,  0,  STATE_START },
    {  +6,  0,  0,  0,  0, 3, -1, 1,  1,   STATE_DATA },
    {  +6,  0,  0,  0,  0, 3, -1, 2,  0,  STATE_START },
    { +14,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    { +14,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    { +15,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_DATA },
    { +15,  0,  0,  0,  0, 4, +1, 0,  0,   STATE_SIGN },
    {  +7,  0,  0,  0,  0, 3, +1, 1,  1,   STATE_DATA },
    {  +7,  0,  0,  0,  0, 3, +1, 2,  0,  STATE_START },
    {  +7,  0,  0,  0,  0, 3, -1, 1,  1,   STATE_DATA },
    {  +7,  0,  0,  0,  0, 3, -1, 2,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  2,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  2,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  3,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, +1, 1,  3,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 2, +1, 2,  2, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, +1, 2,  3, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, +1, 3,  1,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, +1, 4,  0,  STATE_START },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  2,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  2,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  3,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, -1, 1,  3,   STATE_SIGN },
    {  +3,  0,  0,  0,  0, 2, -1, 2,  2, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, -1, 2,  3, STATE_FOLLOW },
    {  +3,  0,  0,  0,  0, 2, -1, 3,  1,   STATE_DATA },
    {  +3,  0,  0,  0,  0, 2, -1, 4,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  4,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  4,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  5,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  5,   STATE_SIGN },
    {  +1, +1,  0,  0,  0, 1, +1, 2,  1,   STATE_DATA },
    {  +1, +1,  0,  0,  0, 1, +1, 3,  0,  STATE_START },
    {  +1, -1,  0,  0,  0, 1, +1, 2,  1,   STATE_DATA },
    {  +1, -1,  0,  0,  0, 1, +1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  6,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  6,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  7,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 1,  7,   STATE_SIGN },
    {  +1, +2,  0,  0,  0, 1, +1, 2,  1,   STATE_DATA },
    {  +1, +2,  0,  0,  0, 1, +1, 3,  0,  STATE_START },
    {  +1, -2,  0,  0,  0, 1, +1, 2,  1,   STATE_DATA },
    {  +1, -2,  0,  0,  0, 1, +1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, +1, 2,  4, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, +1, 2,  5, STATE_FOLLOW },
    {  +1,  0, +1,  0,  0, 1, +1, 3,  0,  STATE_START },
    {  +1,  0, -1,  0,  0, 1, +1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, +1, 2,  6, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, +1, 2,  7, STATE_FOLLOW },
    {  +1,  0, +2,  0,  0, 1, +1, 3,  0,  STATE_START },
    {  +1,  0, -2,  0,  0, 1, +1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, +1, 3,  2,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 3,  2,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, +1, 3,  3,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 3,  3,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, +1, 4,  2, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, +1, 4,  3, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, +1, 5,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, +1, 6,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  4,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  4,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  5,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  5,   STATE_SIGN },
    {  +1, +1,  0,  0,  0, 1, -1, 2,  1,   STATE_DATA },
    {  +1, +1,  0,  0,  0, 1, -1, 3,  0,  STATE_START },
    {  +1, -1,  0,  0,  0, 1, -1, 2,  1,   STATE_DATA },
    {  +1, -1,  0,  0,  0, 1, -1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  6,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  6,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  7,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 1,  7,   STATE_SIGN },
    {  +1, +2,  0,  0,  0, 1, -1, 2,  1,   STATE_DATA },
    {  +1, +2,  0,  0,  0, 1, -1, 3,  0,  STATE_START },
    {  +1, -2,  0,  0,  0, 1, -1, 2,  1,   STATE_DATA },
    {  +1, -2,  0,  0,  0, 1, -1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, -1, 2,  4, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, -1, 2,  5, STATE_FOLLOW },
    {  +1,  0, +1,  0,  0, 1, -1, 3,  0,  STATE_START },
    {  +1,  0, -1,  0,  0, 1, -1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, -1, 2,  6, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, -1, 2,  7, STATE_FOLLOW },
    {  +1,  0, +2,  0,  0, 1, -1, 3,  0,  STATE_START },
    {  +1,  0, -2,  0,  0, 1, -1, 3,  0,  STATE_START },
    {  +1,  0,  0,  0,  0, 1, -1, 3,  2,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 3,  2,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, -1, 3,  3,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 3,  3,   STATE_SIGN },
    {  +1,  0,  0,  0,  0, 1, -1, 4,  2, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, -1, 4,  3, STATE_FOLLOW },
    {  +1,  0,  0,  0,  0, 1, -1, 5,  1,   STATE_DATA },
    {  +1,  0,  0,  0,  0, 1, -1, 6,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 1,  8,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 1,  8,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 1,  9,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 1,  9,   STATE_SIGN },
    {   0, +3,  0,  0,  0, 0, +1, 2,  1,   STATE_DATA },
    {   0, +3,  0,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0, -3,  0,  0,  0, 0, +1, 2,  1,   STATE_DATA },
    {   0, -3,  0,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 1, 10,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 1, 10,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 1, 11,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 1, 11,   STATE_SIGN },
    {   0, +4,  0,  0,  0, 0, +1, 2,  1,   STATE_DATA },
    {   0, +4,  0,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0, -4,  0,  0,  0, 0, +1, 2,  1,   STATE_DATA },
    {   0, -4,  0,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0, +1,  0,  0,  0, 0, +1, 2,  2,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0, +1, 2,  2,   STATE_SIGN },
    {   0, +1,  0,  0,  0, 0, +1, 2,  3,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0, +1, 2,  3,   STATE_SIGN },
    {   0, +1,  0,  0,  0, 0, +1, 3,  2, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0, +1, 3,  3, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0, +1, 5,  0,  STATE_START },
    {   0, -1,  0,  0,  0, 0, +1, 2,  2,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0, +1, 2,  2,   STATE_SIGN },
    {   0, -1,  0,  0,  0, 0, +1, 2,  3,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0, +1, 2,  3,   STATE_SIGN },
    {   0, -1,  0,  0,  0, 0, +1, 3,  2, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0, +1, 3,  3, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 1, 12,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 1, 12,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 1, 13,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 1, 13,   STATE_SIGN },
    {   0, +5,  0,  0,  0, 0, +1, 2,  1,   STATE_DATA },
    {   0, +5,  0,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0, -5,  0,  0,  0, 0, +1, 2,  1,   STATE_DATA },
    {   0, -5,  0,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 1, 14,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 1, 14,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 1, 15,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 1, 15,   STATE_SIGN },
    {   0, +6,  0,  0,  0, 0, +1, 2,  1,   STATE_DATA },
    {   0, +6,  0,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0, -6,  0,  0,  0, 0, +1, 2,  1,   STATE_DATA },
    {   0, -6,  0,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0, +2,  0,  0,  0, 0, +1, 2,  2,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0, +1, 2,  2,   STATE_SIGN },
    {   0, +2,  0,  0,  0, 0, +1, 2,  3,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0, +1, 2,  3,   STATE_SIGN },
    {   0, +2,  0,  0,  0, 0, +1, 3,  2, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0, +1, 3,  3, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0, +1, 5,  0,  STATE_START },
    {   0, -2,  0,  0,  0, 0, +1, 2,  2,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0, +1, 2,  2,   STATE_SIGN },
    {   0, -2,  0,  0,  0, 0, +1, 2,  3,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0, +1, 2,  3,   STATE_SIGN },
    {   0, -2,  0,  0,  0, 0, +1, 3,  2, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0, +1, 3,  3, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 2,  8, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 2,  9, STATE_FOLLOW },
    {   0,  0, +3,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0, -3,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 2, 10, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 2, 11, STATE_FOLLOW },
    {   0,  0, +4,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0, -4,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0, +1,  0,  0, 0, +1, 3,  2, STATE_FOLLOW },
    {   0,  0, +1,  0,  0, 0, +1, 3,  3, STATE_FOLLOW },
    {   0,  0, +1,  0,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0,  0, +1,  0,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0, -1,  0,  0, 0, +1, 3,  2, STATE_FOLLOW },
    {   0,  0, -1,  0,  0, 0, +1, 3,  3, STATE_FOLLOW },
    {   0,  0, -1,  0,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0,  0, -1,  0,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 2, 12, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 2, 13, STATE_FOLLOW },
    {   0,  0, +5,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0, -5,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 2, 14, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 2, 15, STATE_FOLLOW },
    {   0,  0, +6,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0, -6,  0,  0, 0, +1, 3,  0,  STATE_START },
    {   0,  0, +2,  0,  0, 0, +1, 3,  2, STATE_FOLLOW },
    {   0,  0, +2,  0,  0, 0, +1, 3,  3, STATE_FOLLOW },
    {   0,  0, +2,  0,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0,  0, +2,  0,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0, -2,  0,  0, 0, +1, 3,  2, STATE_FOLLOW },
    {   0,  0, -2,  0,  0, 0, +1, 3,  3, STATE_FOLLOW },
    {   0,  0, -2,  0,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0,  0, -2,  0,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 3,  4,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 3,  4,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 3,  5,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 3,  5,   STATE_SIGN },
    {   0,  0,  0, +1,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0,  0,  0, +1,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0, -1,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0,  0,  0, -1,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 3,  6,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 3,  6,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 3,  7,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 3,  7,   STATE_SIGN },
    {   0,  0,  0, +2,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0,  0,  0, +2,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0, -2,  0, 0, +1, 4,  1,   STATE_DATA },
    {   0,  0,  0, -2,  0, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 4,  4, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 4,  5, STATE_FOLLOW },
    {   0,  0,  0,  0, +1, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0, -1, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 4,  6, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 4,  7, STATE_FOLLOW },
    {   0,  0,  0,  0, +2, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0, -2, 0, +1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, +1, 5,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 5,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 5,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 5,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, +1, 6,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 6,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, +1, 7,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, +1, 8,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 1,  8,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 1,  8,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 1,  9,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 1,  9,   STATE_SIGN },
    {   0, +3,  0,  0,  0, 0, -1, 2,  1,   STATE_DATA },
    {   0, +3,  0,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0, -3,  0,  0,  0, 0, -1, 2,  1,   STATE_DATA },
    {   0, -3,  0,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 1, 10,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 1, 10,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 1, 11,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 1, 11,   STATE_SIGN },
    {   0, +4,  0,  0,  0, 0, -1, 2,  1,   STATE_DATA },
    {   0, +4,  0,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0, -4,  0,  0,  0, 0, -1, 2,  1,   STATE_DATA },
    {   0, -4,  0,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0, +1,  0,  0,  0, 0, -1, 2,  2,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0, -1, 2,  2,   STATE_SIGN },
    {   0, +1,  0,  0,  0, 0, -1, 2,  3,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0, -1, 2,  3,   STATE_SIGN },
    {   0, +1,  0,  0,  0, 0, -1, 3,  2, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0, -1, 3,  3, STATE_FOLLOW },
    {   0, +1,  0,  0,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0, +1,  0,  0,  0, 0, -1, 5,  0,  STATE_START },
    {   0, -1,  0,  0,  0, 0, -1, 2,  2,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0, -1, 2,  2,   STATE_SIGN },
    {   0, -1,  0,  0,  0, 0, -1, 2,  3,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0, -1, 2,  3,   STATE_SIGN },
    {   0, -1,  0,  0,  0, 0, -1, 3,  2, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0, -1, 3,  3, STATE_FOLLOW },
    {   0, -1,  0,  0,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0, -1,  0,  0,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 1, 12,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 1, 12,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 1, 13,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 1, 13,   STATE_SIGN },
    {   0, +5,  0,  0,  0, 0, -1, 2,  1,   STATE_DATA },
    {   0, +5,  0,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0, -5,  0,  0,  0, 0, -1, 2,  1,   STATE_DATA },
    {   0, -5,  0,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 1, 14,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 1, 14,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 1, 15,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 1, 15,   STATE_SIGN },
    {   0, +6,  0,  0,  0, 0, -1, 2,  1,   STATE_DATA },
    {   0, +6,  0,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0, -6,  0,  0,  0, 0, -1, 2,  1,   STATE_DATA },
    {   0, -6,  0,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0, +2,  0,  0,  0, 0, -1, 2,  2,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0, -1, 2,  2,   STATE_SIGN },
    {   0, +2,  0,  0,  0, 0, -1, 2,  3,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0, -1, 2,  3,   STATE_SIGN },
    {   0, +2,  0,  0,  0, 0, -1, 3,  2, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0, -1, 3,  3, STATE_FOLLOW },
    {   0, +2,  0,  0,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0, +2,  0,  0,  0, 0, -1, 5,  0,  STATE_START },
    {   0, -2,  0,  0,  0, 0, -1, 2,  2,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0, -1, 2,  2,   STATE_SIGN },
    {   0, -2,  0,  0,  0, 0, -1, 2,  3,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0, -1, 2,  3,   STATE_SIGN },
    {   0, -2,  0,  0,  0, 0, -1, 3,  2, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0, -1, 3,  3, STATE_FOLLOW },
    {   0, -2,  0,  0,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0, -2,  0,  0,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 2,  8, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 2,  9, STATE_FOLLOW },
    {   0,  0, +3,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0, -3,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 2, 10, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 2, 11, STATE_FOLLOW },
    {   0,  0, +4,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0, -4,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0, +1,  0,  0, 0, -1, 3,  2, STATE_FOLLOW },
    {   0,  0, +1,  0,  0, 0, -1, 3,  3, STATE_FOLLOW },
    {   0,  0, +1,  0,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0,  0, +1,  0,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0, -1,  0,  0, 0, -1, 3,  2, STATE_FOLLOW },
    {   0,  0, -1,  0,  0, 0, -1, 3,  3, STATE_FOLLOW },
    {   0,  0, -1,  0,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0,  0, -1,  0,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 2, 12, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 2, 13, STATE_FOLLOW },
    {   0,  0, +5,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0, -5,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 2, 14, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 2, 15, STATE_FOLLOW },
    {   0,  0, +6,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0, -6,  0,  0, 0, -1, 3,  0,  STATE_START },
    {   0,  0, +2,  0,  0, 0, -1, 3,  2, STATE_FOLLOW },
    {   0,  0, +2,  0,  0, 0, -1, 3,  3, STATE_FOLLOW },
    {   0,  0, +2,  0,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0,  0, +2,  0,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0, -2,  0,  0, 0, -1, 3,  2, STATE_FOLLOW },
    {   0,  0, -2,  0,  0, 0, -1, 3,  3, STATE_FOLLOW },
    {   0,  0, -2,  0,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0,  0, -2,  0,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 3,  4,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 3,  4,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 3,  5,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 3,  5,   STATE_SIGN },
    {   0,  0,  0, +1,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0,  0,  0, +1,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0, -1,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0,  0,  0, -1,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 3,  6,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 3,  6,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 3,  7,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 3,  7,   STATE_SIGN },
    {   0,  0,  0, +2,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0,  0,  0, +2,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0, -2,  0, 0, -1, 4,  1,   STATE_DATA },
    {   0,  0,  0, -2,  0, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 4,  4, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 4,  5, STATE_FOLLOW },
    {   0,  0,  0,  0, +1, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0, -1, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 4,  6, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 4,  7, STATE_FOLLOW },
    {   0,  0,  0,  0, +2, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0, -2, 0, -1, 5,  0,  STATE_START },
    {   0,  0,  0,  0,  0, 0, -1, 5,  2,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 5,  2,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 5,  3,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 5,  3,   STATE_SIGN },
    {   0,  0,  0,  0,  0, 0, -1, 6,  2, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 6,  3, STATE_FOLLOW },
    {   0,  0,  0,  0,  0, 0, -1, 7,  1,   STATE_DATA },
    {   0,  0,  0,  0,  0, 0, -1, 8,  0,  STATE_START },
};

#define PROCESS_VALS                                   \
    do {                                               \
        val  <<= lut.val0_bits;                        \
        val   |= lut.val0;                             \
        dst[0] = (val - 1) * lut.sign;                 \
        dst[1] = lut.val1;                             \
        dst[2] = lut.val2;                             \
        dst[3] = lut.val3;                             \
        dst[4] = lut.val4;                             \
        dst[5] = 0;                                    \
        dst[6] = 0;                                    \
        dst[7] = 0;                                    \
        if (lut.num)                                   \
            val = lut.val;                             \
        dst += lut.num;                                \
        if (dst >= last)                               \
            return coeffs;                             \
        lut = ff_dirac_golomb_lut[lut.state + *buf++]; \
    } while (0)

int ff_dirac_golomb_read_16bit(const uint8_t *buf, int bytes,
                               uint8_t *_dst, int coeffs)
{
    LUTState lut = ff_dirac_golomb_lut[*buf++];
    int16_t *dst = (int16_t *)_dst, *last = dst + coeffs;
    int16_t val = 0;

    for (int i = 1; i < bytes; i++)
        PROCESS_VALS;

    /* Reader needs to be flushed */
    PROCESS_VALS;

    /* Still short of coeffs - try to guess and at least output what we have */
    if (lut.state != STATE_START)
        *dst++ = -((lut.state != STATE_SIGN ? (val << 1) | 1 : val) - 1);

    return coeffs - (int)(last - dst);
}

int ff_dirac_golomb_read_32bit(const uint8_t *buf, int bytes,
                               uint8_t *_dst, int coeffs)
{
    LUTState lut = ff_dirac_golomb_lut[*buf++];
    int32_t *dst = (int32_t *)_dst, *last = dst + coeffs;
    int32_t val = 0;

    for (int i = 1; i < bytes; i++)
        PROCESS_VALS;

    /* Reader needs to be flushed */
    PROCESS_VALS;

    /* Still short of coeffs - try to guess and at least output what we have */
    if (lut.state != STATE_START)
        *dst++ = -((lut.state != STATE_SIGN ? (val << 1) | 1 : val) - 1);

    return coeffs - (int)(last - dst);
}
